/**
 * Background class - Improved version
 * 
 * Creates a dynamic cyberpunk background with subtle animations
 * Reduced visual noise for better event visibility
 */
class Background {
  // Background properties
  color bgColor;
  color accentColor1;
  color accentColor2;
  
  // Movement properties
  float noiseScale = 0.01;  // Reduced from 0.02 for subtler pattern
  float noiseOffset = 0;
  float noiseSpeed = 0.003;  // Reduced from 0.005 for slower movement
  
  // Timing
  float cps = 0.5;
  float lastCycleTime = 0;
  
  Background() {
    // Darker cyberpunk colors for better contrast with events
    bgColor = color(5, 7, 12);  // Darker than original
    accentColor1 = color(0, 50, 80);  // More subtle
    accentColor2 = color(40, 0, 60);  // More subtle
  }
  
  void update() {
    // Update noise movement
    noiseOffset += noiseSpeed;
    
    // Check for cycle changes
    float currentCycleTime = millis() / 1000.0 * cps;
    if (floor(currentCycleTime) > floor(lastCycleTime)) {
      // New cycle - could trigger effects here
    }
    lastCycleTime = currentCycleTime;
  }
  
  void display() {
    // Create gradient background
    noiseDetail(4, 0.4);  // Simplified noise
    
    // Fill with base color - more transparent for more visible motion trails
    noStroke();
    fill(bgColor, 30);  // Reduced from 40
    rect(0, 0, width, height);
    
    // Add subtle noise pattern
    float motionFactor = sin(millis() * 0.0005) * 0.5 + 0.5;  // Slowed from 0.001
    
    // Reduced noise pattern density - iterate every 8 pixels instead of 4
    for (int y = 0; y < height; y += 8) {
      for (int x = 0; x < width; x += 8) {
        float noiseVal = noise(x * noiseScale, y * noiseScale, noiseOffset);
        
        // Only draw higher noise values (reduced density)
        if (noiseVal > 0.8) {  // Increased from 0.7
          color pixelColor;
          
          // Create different zones
          if (noiseVal > 0.9) {  // Increased from 0.85
            // Highlight areas
            pixelColor = lerpColor(accentColor1, accentColor2, 
                                   sin(x * 0.005 + millis() * 0.0002) * 0.5 + 0.5);
            pixelColor = color(red(pixelColor), green(pixelColor), blue(pixelColor), 
                              15 + 10 * motionFactor);  // Reduced from 20+20
          } else {
            // Subtle accent
            pixelColor = lerpColor(bgColor, accentColor1, 0.2);  // Reduced from 0.3
            pixelColor = color(red(pixelColor), green(pixelColor), blue(pixelColor), 8);  // Reduced from 10
          }
          
          // Draw 8x8 pixel block for better performance and subtler texture
          fill(pixelColor);
          rect(x, y, 8, 8);
        }
      }
    }
    
    // Draw horizontal scan lines - fewer and more subtle
    drawScanlines();
  }
  
  void drawScanlines() {
    // Fewer scan lines (every 8 pixels instead of 4)
    stroke(255, 5);  // Reduced from 8
    strokeWeight(1);
    for (int y = 0; y < height; y += 8) {
      line(0, y, width, y);
    }
    
    // Draw brighter scanline that moves
    float movingScanline = (millis() % 8000) / 8000.0 * height;  // Slower movement (5000 to 8000)
    stroke(255, 10);  // Reduced from 15
    strokeWeight(2);
    line(0, movingScanline, width, movingScanline);
  }
  
  void setCPS(float newCps) {
    this.cps = newCps;
  }
}
